package de.kuschku.util.observables.lists;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v7.util.SortedList;

import com.afollestad.materialdialogs.MaterialDialog;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;

import de.kuschku.util.backports.Stream;
import de.kuschku.util.observables.callbacks.UICallback;
import de.kuschku.util.observables.callbacks.wrappers.MultiUICallbackWrapper;

import static de.kuschku.util.AndroidAssert.assertTrue;

public class ObservableSortedList<T> implements IObservableList<UICallback, T> {
    @NonNull
    private final SortedList<T> list;
    private final boolean reverse;

    @NonNull
    private final MultiUICallbackWrapper callback = MultiUICallbackWrapper.of();
    @NonNull
    private ItemComparator<T> comparator;

    public ObservableSortedList(@NonNull Class<T> cl, @NonNull ItemComparator<T> comparator) {
        this(cl, comparator, false);
    }

    public ObservableSortedList(@NonNull Class<T> cl, @NonNull ItemComparator<T> comparator, boolean reverse) {
        this.list = new SortedList<>(cl, new Callback());
        this.comparator = comparator;
        this.reverse = reverse;
    }

    @Override
    public void addCallback(@NonNull UICallback callback) {
        this.callback.addCallback(callback);
    }

    @Override
    public void removeCallback(@NonNull UICallback callback) {
        this.callback.removeCallback(callback);
    }

    public void setComparator(@NonNull ItemComparator<T> comparator) {
        this.comparator = comparator;
    }

    @Nullable
    public T last() {
        if (list.size() == 0) return null;

        return list.get(list.size() - 1);
    }

    @Override
    public void add(int location, T object) {
        list.add(object);
    }

    @Override
    public boolean add(T object) {
        list.add(object);
        return true;
    }

    @Override
    public boolean addAll(int location, @NonNull Collection<? extends T> collection) {
        list.addAll((Collection<T>) collection);
        return true;
    }

    @Override
    public boolean addAll(@NonNull Collection<? extends T> collection) {
        list.addAll((Collection<T>) collection);
        return false;
    }

    @Override
    public void clear() {
        list.clear();
    }

    @Override
    public boolean contains(Object object) {
        return indexOf(object) != SortedList.INVALID_POSITION;
    }

    @Override
    public boolean containsAll(@NonNull Collection<?> collection) {
        return new Stream<>(collection).allMatch(this::contains);
    }

    @Override
    public T get(int location) {
        return list.get(location);
    }

    @Override
    public int indexOf(Object object) {
        return list.indexOf((T) object);
    }

    @Override
    public boolean isEmpty() {
        return list.size() == 0;
    }

    @NonNull
    @Override
    public Iterator<T> iterator() {
        return new CallbackedSortedListIterator();
    }

    @Override
    public int lastIndexOf(Object object) {
        return indexOf(object);
    }

    @NonNull
    @Override
    public ListIterator<T> listIterator() {
        return new CallbackedSortedListIterator();
    }

    @NonNull
    @Override
    public ListIterator<T> listIterator(int location) {
        return new CallbackedSortedListIterator(location);
    }

    @Nullable
    @Override
    public T remove(int location) {
        T item = list.get(location);
        list.remove(item);
        return item;
    }

    @Override
    public boolean remove(Object object) {
        try {
            list.remove((T) object);
            return true;
        } catch (ClassCastException e) {
            return false;
        }
    }

    @Override
    public boolean removeAll(@NonNull Collection<?> collection) {
        boolean result = true;
        for (Object o : collection) {
            result &= remove(o);
        }
        return result;
    }

    @Override
    public boolean retainAll(@NonNull Collection<?> collection) {
        return false;
    }

    @Nullable
    @Override
    public T set(int location, T object) {
        return null;
    }

    @Override
    public int size() {
        return list.size();
    }

    @NonNull
    @Override
    public List<T> subList(int start, int end) {
        assertTrue(start <= end);
        assertTrue(start >= 0);
        assertTrue(end <= list.size());

        List<T> subList = new ArrayList<>(end - start);
        for (int i = start; i < end; i++) {
            subList.add(list.get(i));
        }
        return subList;
    }

    @NonNull
    @Override
    public Object[] toArray() {
        throw new MaterialDialog.NotImplementedException("Not implemented");
    }

    @NonNull
    @Override
    public <T1> T1[] toArray(@NonNull T1[] array) {
        throw new MaterialDialog.NotImplementedException("Not implemented");
    }

    public void notifyItemChanged(int position) {
        callback.notifyItemChanged(position);
    }

    public interface ItemComparator<T> {
        int compare(T o1, T o2);

        boolean areContentsTheSame(T oldItem, T newItem);

        boolean areItemsTheSame(T item1, T item2);
    }

    class Callback extends SortedList.Callback<T> {
        @Override
        public int compare(T o1, T o2) {
            return (reverse) ? comparator.compare(o2, o1) : comparator.compare(o1, o2);
        }

        @Override
        public void onInserted(int position, int count) {
            if (callback != null)
                if (count == 1)
                    callback.notifyItemInserted(position);
                else
                    callback.notifyItemRangeInserted(position, count);
        }

        @Override
        public void onRemoved(int position, int count) {
            if (callback != null)
                if (count == 1)
                    callback.notifyItemRemoved(position);
                else
                    callback.notifyItemRangeRemoved(position, count);
        }

        @Override
        public void onMoved(int fromPosition, int toPosition) {
            if (callback != null)
                callback.notifyItemMoved(fromPosition, toPosition);
        }

        @Override
        public void onChanged(int position, int count) {
            if (callback != null)
                if (count == 1)
                    callback.notifyItemChanged(position);
                else
                    callback.notifyItemRangeChanged(position, count);
        }

        @Override
        public boolean areContentsTheSame(T oldItem, T newItem) {
            return comparator.areContentsTheSame(oldItem, newItem);
        }

        @Override
        public boolean areItemsTheSame(T item1, T item2) {
            return comparator.areItemsTheSame(item1, item2);
        }
    }

    class CallbackedSortedListIterator implements Iterator<T>, ListIterator<T> {
        int position;

        public CallbackedSortedListIterator() {
            this(0);
        }

        public CallbackedSortedListIterator(int position) {
            this.position = position;
        }

        @Override
        public void add(T object) {
            list.add(object);
        }

        @Override
        public boolean hasNext() {
            return list.size() > position + 1;
        }

        @Override
        public boolean hasPrevious() {
            return false;
        }

        @Override
        public T next() {
            return list.get(position++);
        }

        @Override
        public int nextIndex() {
            return position + 1;
        }

        @Override
        public T previous() {
            return list.get(position--);
        }

        @Override
        public int previousIndex() {
            return position - 1;
        }

        @Override
        public void remove() {
            list.remove(list.get(position));
            callback.notifyItemRemoved(position);
        }

        @Override
        public void set(T object) {
            list.remove(list.get(position));
            list.add(object);
        }
    }
}
