use ffmpeg_dev::sys as ffi;
use enum_primitive::*;

enum_from_primitive! {
    #[doc = " Pixel format."]
    #[doc = ""]
    #[doc = " @note"]
    #[doc = " AV_PIX_FMT_RGB32 is handled in an endian-specific manner. An RGBA"]
    #[doc = " color is put together as:"]
    #[doc = "  (A << 24) | (R << 16) | (G << 8) | B"]
    #[doc = " This is stored as BGRA on little-endian CPU architectures and ARGB on"]
    #[doc = " big-endian CPUs."]
    #[doc = ""]
    #[doc = " @note"]
    #[doc = " If the resolution is not a multiple of the chroma subsampling factor"]
    #[doc = " then the chroma plane resolution must be rounded up."]
    #[doc = ""]
    #[doc = " @par"]
    #[doc = " When the pixel format is palettized RGB32 (AV_PIX_FMT_PAL8), the palettized"]
    #[doc = " image data is stored in AVFrame.data[0]. The palette is transported in"]
    #[doc = " AVFrame.data[1], is 1024 bytes long (256 4-byte entries) and is"]
    #[doc = " formatted the same as in AV_PIX_FMT_RGB32 described above (i.e., it is"]
    #[doc = " also endian-specific). Note also that the individual RGB32 palette"]
    #[doc = " components stored in AVFrame.data[1] should be in the range 0..255."]
    #[doc = " This is important as many custom PAL8 video codecs that were designed"]
    #[doc = " to run on the IBM VGA graphics adapter use 6-bit palette components."]
    #[doc = ""]
    #[doc = " @par"]
    #[doc = " For all the 8 bits per pixel formats, an RGB32 palette is in data[1] like"]
    #[doc = " for pal8. This palette is filled in automatically by the function"]
    #[doc = " allocating the picture."]
    #[allow(non_camel_case_types)]
    #[derive(Debug, Copy, Clone, PartialEq)]
    #[repr(i32)]
    pub enum AVPixelFormat {
        NONE = ffi::AVPixelFormat_AV_PIX_FMT_NONE,
        #[doc = "< planar YUV 4:2:0, 12bpp, (1 Cr & Cb sample per 2x2 Y samples)"]
        YUV420P = ffi::AVPixelFormat_AV_PIX_FMT_YUV420P,
        #[doc = "< packed YUV 4:2:2, 16bpp, Y0 Cb Y1 Cr"]
        YUYV422 = ffi::AVPixelFormat_AV_PIX_FMT_YUYV422,
        #[doc = "< packed RGB 8:8:8, 24bpp, RGBRGB..."]
        RGB24 = ffi::AVPixelFormat_AV_PIX_FMT_RGB24,
        #[doc = "< packed RGB 8:8:8, 24bpp, BGRBGR..."]
        BGR24 = ffi::AVPixelFormat_AV_PIX_FMT_BGR24,
        #[doc = "< planar YUV 4:2:2, 16bpp, (1 Cr & Cb sample per 2x1 Y samples)"]
        YUV422P = ffi::AVPixelFormat_AV_PIX_FMT_YUV422P,
        #[doc = "< planar YUV 4:4:4, 24bpp, (1 Cr & Cb sample per 1x1 Y samples)"]
        YUV444P = ffi::AVPixelFormat_AV_PIX_FMT_YUV444P,
        #[doc = "< planar YUV 4:1:0,  9bpp, (1 Cr & Cb sample per 4x4 Y samples)"]
        YUV410P = ffi::AVPixelFormat_AV_PIX_FMT_YUV410P,
        #[doc = "< planar YUV 4:1:1, 12bpp, (1 Cr & Cb sample per 4x1 Y samples)"]
        YUV411P = ffi::AVPixelFormat_AV_PIX_FMT_YUV411P,
        #[doc = "<        Y        ,  8bpp"]
        GRAY8 = ffi::AVPixelFormat_AV_PIX_FMT_GRAY8,
        #[doc = "<        Y        ,  1bpp, 0 is white, 1 is black, in each byte pixels are ordered from the msb to the lsb"]
        MONOWHITE = ffi::AVPixelFormat_AV_PIX_FMT_MONOWHITE,
        #[doc = "<        Y        ,  1bpp, 0 is black, 1 is white, in each byte pixels are ordered from the msb to the lsb"]
        MONOBLACK = ffi::AVPixelFormat_AV_PIX_FMT_MONOBLACK,
        #[doc = "< 8 bits with AV_PIX_FMT_RGB32 palette"]
        PAL8 = ffi::AVPixelFormat_AV_PIX_FMT_PAL8,
        #[doc = "< planar YUV 4:2:0, 12bpp, full scale (JPEG), deprecated in favor of AV_PIX_FMT_YUV420P and setting color_range"]
        YUVJ420P = ffi::AVPixelFormat_AV_PIX_FMT_YUVJ420P,
        #[doc = "< planar YUV 4:2:2, 16bpp, full scale (JPEG), deprecated in favor of AV_PIX_FMT_YUV422P and setting color_range"]
        YUVJ422P = ffi::AVPixelFormat_AV_PIX_FMT_YUVJ422P,
        #[doc = "< planar YUV 4:4:4, 24bpp, full scale (JPEG), deprecated in favor of AV_PIX_FMT_YUV444P and setting color_range"]
        YUVJ444P = ffi::AVPixelFormat_AV_PIX_FMT_YUVJ444P,
        #[doc = "< packed YUV 4:2:2, 16bpp, Cb Y0 Cr Y1"]
        UYVY422 = ffi::AVPixelFormat_AV_PIX_FMT_UYVY422,
        #[doc = "< packed YUV 4:1:1, 12bpp, Cb Y0 Y1 Cr Y2 Y3"]
        UYYVYY411 = ffi::AVPixelFormat_AV_PIX_FMT_UYYVYY411,
        #[doc = "< packed RGB 3:3:2,  8bpp, (msb)2B 3G 3R(lsb)"]
        BGR8 = ffi::AVPixelFormat_AV_PIX_FMT_BGR8,
        #[doc = "< packed RGB 1:2:1 bitstream,  4bpp, (msb)1B 2G 1R(lsb), a byte contains two pixels, the first pixel in the byte is the one composed by the 4 msb bits"]
        BGR4 = ffi::AVPixelFormat_AV_PIX_FMT_BGR4,
        #[doc = "< packed RGB 1:2:1,  8bpp, (msb)1B 2G 1R(lsb)"]
        BGR4_BYTE = ffi::AVPixelFormat_AV_PIX_FMT_BGR4_BYTE,
        #[doc = "< packed RGB 3:3:2,  8bpp, (msb)2R 3G 3B(lsb)"]
        RGB8 = ffi::AVPixelFormat_AV_PIX_FMT_RGB8,
        #[doc = "< packed RGB 1:2:1 bitstream,  4bpp, (msb)1R 2G 1B(lsb), a byte contains two pixels, the first pixel in the byte is the one composed by the 4 msb bits"]
        RGB4 = ffi::AVPixelFormat_AV_PIX_FMT_RGB4,
        #[doc = "< packed RGB 1:2:1,  8bpp, (msb)1R 2G 1B(lsb)"]
        RGB4_BYTE = ffi::AVPixelFormat_AV_PIX_FMT_RGB4_BYTE,
        #[doc = "< planar YUV 4:2:0, 12bpp, 1 plane for Y and 1 plane for the UV components, which are interleaved (first byte U and the following byte V)"]
        NV12 = ffi::AVPixelFormat_AV_PIX_FMT_NV12,
        #[doc = "< as above, but U and V bytes are swapped"]
        NV21 = ffi::AVPixelFormat_AV_PIX_FMT_NV21,
        #[doc = "< packed ARGB 8:8:8:8, 32bpp, ARGBARGB..."]
        ARGB = ffi::AVPixelFormat_AV_PIX_FMT_ARGB,
        #[doc = "< packed RGBA 8:8:8:8, 32bpp, RGBARGBA..."]
        RGBA = ffi::AVPixelFormat_AV_PIX_FMT_RGBA,
        #[doc = "< packed ABGR 8:8:8:8, 32bpp, ABGRABGR..."]
        ABGR = ffi::AVPixelFormat_AV_PIX_FMT_ABGR,
        #[doc = "< packed BGRA 8:8:8:8, 32bpp, BGRABGRA..."]
        BGRA = ffi::AVPixelFormat_AV_PIX_FMT_BGRA,
        #[doc = "<        Y        , 16bpp, big-endian"]
        GRAY16BE = ffi::AVPixelFormat_AV_PIX_FMT_GRAY16BE,
        #[doc = "<        Y        , 16bpp, little-endian"]
        GRAY16LE = ffi::AVPixelFormat_AV_PIX_FMT_GRAY16LE,
        #[doc = "< planar YUV 4:4:0 (1 Cr & Cb sample per 1x2 Y samples)"]
        YUV440P = ffi::AVPixelFormat_AV_PIX_FMT_YUV440P,
        #[doc = "< planar YUV 4:4:0 full scale (JPEG), deprecated in favor of AV_PIX_FMT_YUV440P and setting color_range"]
        YUVJ440P = ffi::AVPixelFormat_AV_PIX_FMT_YUVJ440P,
        #[doc = "< planar YUV 4:2:0, 20bpp, (1 Cr & Cb sample per 2x2 Y & A samples)"]
        YUVA420P = ffi::AVPixelFormat_AV_PIX_FMT_YUVA420P,
        #[doc = "< packed RGB 16:16:16, 48bpp, 16R, 16G, 16B, the 2-byte value for each R/G/B component is stored as big-endian"]
        RGB48BE = ffi::AVPixelFormat_AV_PIX_FMT_RGB48BE,
        #[doc = "< packed RGB 16:16:16, 48bpp, 16R, 16G, 16B, the 2-byte value for each R/G/B component is stored as little-endian"]
        RGB48LE = ffi::AVPixelFormat_AV_PIX_FMT_RGB48LE,
        #[doc = "< packed RGB 5:6:5, 16bpp, (msb)   5R 6G 5B(lsb), big-endian"]
        RGB565BE = ffi::AVPixelFormat_AV_PIX_FMT_RGB565BE,
        #[doc = "< packed RGB 5:6:5, 16bpp, (msb)   5R 6G 5B(lsb), little-endian"]
        RGB565LE = ffi::AVPixelFormat_AV_PIX_FMT_RGB565LE,
        #[doc = "< packed RGB 5:5:5, 16bpp, (msb)1X 5R 5G 5B(lsb), big-endian   , X=unused/undefined"]
        RGB555BE = ffi::AVPixelFormat_AV_PIX_FMT_RGB555BE,
        #[doc = "< packed RGB 5:5:5, 16bpp, (msb)1X 5R 5G 5B(lsb), little-endian, X=unused/undefined"]
        RGB555LE = ffi::AVPixelFormat_AV_PIX_FMT_RGB555LE,
        #[doc = "< packed BGR 5:6:5, 16bpp, (msb)   5B 6G 5R(lsb), big-endian"]
        BGR565BE = ffi::AVPixelFormat_AV_PIX_FMT_BGR565BE,
        #[doc = "< packed BGR 5:6:5, 16bpp, (msb)   5B 6G 5R(lsb), little-endian"]
        BGR565LE = ffi::AVPixelFormat_AV_PIX_FMT_BGR565LE,
        #[doc = "< packed BGR 5:5:5, 16bpp, (msb)1X 5B 5G 5R(lsb), big-endian   , X=unused/undefined"]
        BGR555BE = ffi::AVPixelFormat_AV_PIX_FMT_BGR555BE,
        #[doc = "< packed BGR 5:5:5, 16bpp, (msb)1X 5B 5G 5R(lsb), little-endian, X=unused/undefined"]
        BGR555LE = ffi::AVPixelFormat_AV_PIX_FMT_BGR555LE,
        #[doc = "< HW acceleration through VA API at motion compensation entry-point, Picture.data[3] contains a vaapi_render_state struct which contains macroblocks as well as various fields extracted from headers"]
        VAAPI_MOCO = ffi::AVPixelFormat_AV_PIX_FMT_VAAPI_MOCO,
        #[doc = "< HW acceleration through VA API at IDCT entry-point, Picture.data[3] contains a vaapi_render_state struct which contains fields extracted from headers"]
        VAAPI_IDCT = ffi::AVPixelFormat_AV_PIX_FMT_VAAPI_IDCT,
        #[doc = "< HW decoding through VA API, Picture.data[3] contains a VASurfaceID"]
        VAAPI_VLD = ffi::AVPixelFormat_AV_PIX_FMT_VAAPI_VLD,
        #[doc = "< planar YUV 4:2:0, 24bpp, (1 Cr & Cb sample per 2x2 Y samples), little-endian"]
        YUV420P16LE = ffi::AVPixelFormat_AV_PIX_FMT_YUV420P16LE,
        #[doc = "< planar YUV 4:2:0, 24bpp, (1 Cr & Cb sample per 2x2 Y samples), big-endian"]
        YUV420P16BE = ffi::AVPixelFormat_AV_PIX_FMT_YUV420P16BE,
        #[doc = "< planar YUV 4:2:2, 32bpp, (1 Cr & Cb sample per 2x1 Y samples), little-endian"]
        YUV422P16LE = ffi::AVPixelFormat_AV_PIX_FMT_YUV422P16LE,
        #[doc = "< planar YUV 4:2:2, 32bpp, (1 Cr & Cb sample per 2x1 Y samples), big-endian"]
        YUV422P16BE = ffi::AVPixelFormat_AV_PIX_FMT_YUV422P16BE,
        #[doc = "< planar YUV 4:4:4, 48bpp, (1 Cr & Cb sample per 1x1 Y samples), little-endian"]
        YUV444P16LE = ffi::AVPixelFormat_AV_PIX_FMT_YUV444P16LE,
        #[doc = "< planar YUV 4:4:4, 48bpp, (1 Cr & Cb sample per 1x1 Y samples), big-endian"]
        YUV444P16BE = ffi::AVPixelFormat_AV_PIX_FMT_YUV444P16BE,
        #[doc = "< HW decoding through DXVA2, Picture.data[3] contains a LPDIRECT3DSURFACE9 pointer"]
        DXVA2_VLD = ffi::AVPixelFormat_AV_PIX_FMT_DXVA2_VLD,
        #[doc = "< packed RGB 4:4:4, 16bpp, (msb)4X 4R 4G 4B(lsb), little-endian, X=unused/undefined"]
        RGB444LE = ffi::AVPixelFormat_AV_PIX_FMT_RGB444LE,
        #[doc = "< packed RGB 4:4:4, 16bpp, (msb)4X 4R 4G 4B(lsb), big-endian,    X=unused/undefined"]
        RGB444BE = ffi::AVPixelFormat_AV_PIX_FMT_RGB444BE,
        #[doc = "< packed BGR 4:4:4, 16bpp, (msb)4X 4B 4G 4R(lsb), little-endian, X=unused/undefined"]
        BGR444LE = ffi::AVPixelFormat_AV_PIX_FMT_BGR444LE,
        #[doc = "< packed BGR 4:4:4, 16bpp, (msb)4X 4B 4G 4R(lsb), big-endian,    X=unused/undefined"]
        BGR444BE = ffi::AVPixelFormat_AV_PIX_FMT_BGR444BE,
        #[doc = "< 8 bits gray, 8 bits alpha"]
        YA8 = ffi::AVPixelFormat_AV_PIX_FMT_YA8,
        #[doc = "< packed RGB 16:16:16, 48bpp, 16B, 16G, 16R, the 2-byte value for each R/G/B component is stored as big-endian"]
        BGR48BE = ffi::AVPixelFormat_AV_PIX_FMT_BGR48BE,
        #[doc = "< packed RGB 16:16:16, 48bpp, 16B, 16G, 16R, the 2-byte value for each R/G/B component is stored as little-endian"]
        BGR48LE = ffi::AVPixelFormat_AV_PIX_FMT_BGR48LE,
        #[doc = "< planar YUV 4:2:0, 13.5bpp, (1 Cr & Cb sample per 2x2 Y samples), big-endian"]
        YUV420P9BE = ffi::AVPixelFormat_AV_PIX_FMT_YUV420P9BE,
        #[doc = "< planar YUV 4:2:0, 13.5bpp, (1 Cr & Cb sample per 2x2 Y samples), little-endian"]
        YUV420P9LE = ffi::AVPixelFormat_AV_PIX_FMT_YUV420P9LE,
        #[doc = "< planar YUV 4:2:0, 15bpp, (1 Cr & Cb sample per 2x2 Y samples), big-endian"]
        YUV420P10BE = ffi::AVPixelFormat_AV_PIX_FMT_YUV420P10BE,
        #[doc = "< planar YUV 4:2:0, 15bpp, (1 Cr & Cb sample per 2x2 Y samples), little-endian"]
        YUV420P10LE = ffi::AVPixelFormat_AV_PIX_FMT_YUV420P10LE,
        #[doc = "< planar YUV 4:2:2, 20bpp, (1 Cr & Cb sample per 2x1 Y samples), big-endian"]
        YUV422P10BE = ffi::AVPixelFormat_AV_PIX_FMT_YUV422P10BE,
        #[doc = "< planar YUV 4:2:2, 20bpp, (1 Cr & Cb sample per 2x1 Y samples), little-endian"]
        YUV422P10LE = ffi::AVPixelFormat_AV_PIX_FMT_YUV422P10LE,
        #[doc = "< planar YUV 4:4:4, 27bpp, (1 Cr & Cb sample per 1x1 Y samples), big-endian"]
        YUV444P9BE = ffi::AVPixelFormat_AV_PIX_FMT_YUV444P9BE,
        #[doc = "< planar YUV 4:4:4, 27bpp, (1 Cr & Cb sample per 1x1 Y samples), little-endian"]
        YUV444P9LE = ffi::AVPixelFormat_AV_PIX_FMT_YUV444P9LE,
        #[doc = "< planar YUV 4:4:4, 30bpp, (1 Cr & Cb sample per 1x1 Y samples), big-endian"]
        YUV444P10BE = ffi::AVPixelFormat_AV_PIX_FMT_YUV444P10BE,
        #[doc = "< planar YUV 4:4:4, 30bpp, (1 Cr & Cb sample per 1x1 Y samples), little-endian"]
        YUV444P10LE = ffi::AVPixelFormat_AV_PIX_FMT_YUV444P10LE,
        #[doc = "< planar YUV 4:2:2, 18bpp, (1 Cr & Cb sample per 2x1 Y samples), big-endian"]
        YUV422P9BE = ffi::AVPixelFormat_AV_PIX_FMT_YUV422P9BE,
        #[doc = "< planar YUV 4:2:2, 18bpp, (1 Cr & Cb sample per 2x1 Y samples), little-endian"]
        YUV422P9LE = ffi::AVPixelFormat_AV_PIX_FMT_YUV422P9LE,
        #[doc = "< planar GBR 4:4:4 24bpp"]
        GBRP = ffi::AVPixelFormat_AV_PIX_FMT_GBRP,
        #[doc = "< planar GBR 4:4:4 27bpp, big-endian"]
        GBRP9BE = ffi::AVPixelFormat_AV_PIX_FMT_GBRP9BE,
        #[doc = "< planar GBR 4:4:4 27bpp, little-endian"]
        GBRP9LE = ffi::AVPixelFormat_AV_PIX_FMT_GBRP9LE,
        #[doc = "< planar GBR 4:4:4 30bpp, big-endian"]
        GBRP10BE = ffi::AVPixelFormat_AV_PIX_FMT_GBRP10BE,
        #[doc = "< planar GBR 4:4:4 30bpp, little-endian"]
        GBRP10LE = ffi::AVPixelFormat_AV_PIX_FMT_GBRP10LE,
        #[doc = "< planar GBR 4:4:4 48bpp, big-endian"]
        GBRP16BE = ffi::AVPixelFormat_AV_PIX_FMT_GBRP16BE,
        #[doc = "< planar GBR 4:4:4 48bpp, little-endian"]
        GBRP16LE = ffi::AVPixelFormat_AV_PIX_FMT_GBRP16LE,
        #[doc = "< planar YUV 4:2:2 24bpp, (1 Cr & Cb sample per 2x1 Y & A samples)"]
        YUVA422P = ffi::AVPixelFormat_AV_PIX_FMT_YUVA422P,
        #[doc = "< planar YUV 4:4:4 32bpp, (1 Cr & Cb sample per 1x1 Y & A samples)"]
        YUVA444P = ffi::AVPixelFormat_AV_PIX_FMT_YUVA444P,
        #[doc = "< planar YUV 4:2:0 22.5bpp, (1 Cr & Cb sample per 2x2 Y & A samples), big-endian"]
        YUVA420P9BE = ffi::AVPixelFormat_AV_PIX_FMT_YUVA420P9BE,
        #[doc = "< planar YUV 4:2:0 22.5bpp, (1 Cr & Cb sample per 2x2 Y & A samples), little-endian"]
        YUVA420P9LE = ffi::AVPixelFormat_AV_PIX_FMT_YUVA420P9LE,
        #[doc = "< planar YUV 4:2:2 27bpp, (1 Cr & Cb sample per 2x1 Y & A samples), big-endian"]
        YUVA422P9BE = ffi::AVPixelFormat_AV_PIX_FMT_YUVA422P9BE,
        #[doc = "< planar YUV 4:2:2 27bpp, (1 Cr & Cb sample per 2x1 Y & A samples), little-endian"]
        YUVA422P9LE = ffi::AVPixelFormat_AV_PIX_FMT_YUVA422P9LE,
        #[doc = "< planar YUV 4:4:4 36bpp, (1 Cr & Cb sample per 1x1 Y & A samples), big-endian"]
        YUVA444P9BE = ffi::AVPixelFormat_AV_PIX_FMT_YUVA444P9BE,
        #[doc = "< planar YUV 4:4:4 36bpp, (1 Cr & Cb sample per 1x1 Y & A samples), little-endian"]
        YUVA444P9LE = ffi::AVPixelFormat_AV_PIX_FMT_YUVA444P9LE,
        #[doc = "< planar YUV 4:2:0 25bpp, (1 Cr & Cb sample per 2x2 Y & A samples, big-endian)"]
        YUVA420P10BE = ffi::AVPixelFormat_AV_PIX_FMT_YUVA420P10BE,
        #[doc = "< planar YUV 4:2:0 25bpp, (1 Cr & Cb sample per 2x2 Y & A samples, little-endian)"]
        YUVA420P10LE = ffi::AVPixelFormat_AV_PIX_FMT_YUVA420P10LE,
        #[doc = "< planar YUV 4:2:2 30bpp, (1 Cr & Cb sample per 2x1 Y & A samples, big-endian)"]
        YUVA422P10BE = ffi::AVPixelFormat_AV_PIX_FMT_YUVA422P10BE,
        #[doc = "< planar YUV 4:2:2 30bpp, (1 Cr & Cb sample per 2x1 Y & A samples, little-endian)"]
        YUVA422P10LE = ffi::AVPixelFormat_AV_PIX_FMT_YUVA422P10LE,
        #[doc = "< planar YUV 4:4:4 40bpp, (1 Cr & Cb sample per 1x1 Y & A samples, big-endian)"]
        YUVA444P10BE = ffi::AVPixelFormat_AV_PIX_FMT_YUVA444P10BE,
        #[doc = "< planar YUV 4:4:4 40bpp, (1 Cr & Cb sample per 1x1 Y & A samples, little-endian)"]
        YUVA444P10LE = ffi::AVPixelFormat_AV_PIX_FMT_YUVA444P10LE,
        #[doc = "< planar YUV 4:2:0 40bpp, (1 Cr & Cb sample per 2x2 Y & A samples, big-endian)"]
        YUVA420P16BE = ffi::AVPixelFormat_AV_PIX_FMT_YUVA420P16BE,
        #[doc = "< planar YUV 4:2:0 40bpp, (1 Cr & Cb sample per 2x2 Y & A samples, little-endian)"]
        YUVA420P16LE = ffi::AVPixelFormat_AV_PIX_FMT_YUVA420P16LE,
        #[doc = "< planar YUV 4:2:2 48bpp, (1 Cr & Cb sample per 2x1 Y & A samples, big-endian)"]
        YUVA422P16BE = ffi::AVPixelFormat_AV_PIX_FMT_YUVA422P16BE,
        #[doc = "< planar YUV 4:2:2 48bpp, (1 Cr & Cb sample per 2x1 Y & A samples, little-endian)"]
        YUVA422P16LE = ffi::AVPixelFormat_AV_PIX_FMT_YUVA422P16LE,
        #[doc = "< planar YUV 4:4:4 64bpp, (1 Cr & Cb sample per 1x1 Y & A samples, big-endian)"]
        YUVA444P16BE = ffi::AVPixelFormat_AV_PIX_FMT_YUVA444P16BE,
        #[doc = "< planar YUV 4:4:4 64bpp, (1 Cr & Cb sample per 1x1 Y & A samples, little-endian)"]
        YUVA444P16LE = ffi::AVPixelFormat_AV_PIX_FMT_YUVA444P16LE,
        #[doc = "< HW acceleration through VDPAU, Picture.data[3] contains a VdpVideoSurface"]
        VDPAU = ffi::AVPixelFormat_AV_PIX_FMT_VDPAU,
        #[doc = "< packed XYZ 4:4:4, 36 bpp, (msb) 12X, 12Y, 12Z (lsb), the 2-byte value for each X/Y/Z is stored as little-endian, the 4 lower bits are set to 0"]
        XYZ12LE = ffi::AVPixelFormat_AV_PIX_FMT_XYZ12LE,
        #[doc = "< packed XYZ 4:4:4, 36 bpp, (msb) 12X, 12Y, 12Z (lsb), the 2-byte value for each X/Y/Z is stored as big-endian, the 4 lower bits are set to 0"]
        XYZ12BE = ffi::AVPixelFormat_AV_PIX_FMT_XYZ12BE,
        #[doc = "< interleaved chroma YUV 4:2:2, 16bpp, (1 Cr & Cb sample per 2x1 Y samples)"]
        NV16 = ffi::AVPixelFormat_AV_PIX_FMT_NV16,
        #[doc = "< interleaved chroma YUV 4:2:2, 20bpp, (1 Cr & Cb sample per 2x1 Y samples), little-endian"]
        NV20LE = ffi::AVPixelFormat_AV_PIX_FMT_NV20LE,
        #[doc = "< interleaved chroma YUV 4:2:2, 20bpp, (1 Cr & Cb sample per 2x1 Y samples), big-endian"]
        NV20BE = ffi::AVPixelFormat_AV_PIX_FMT_NV20BE,
        #[doc = "< packed RGBA 16:16:16:16, 64bpp, 16R, 16G, 16B, 16A, the 2-byte value for each R/G/B/A component is stored as big-endian"]
        RGBA64BE = ffi::AVPixelFormat_AV_PIX_FMT_RGBA64BE,
        #[doc = "< packed RGBA 16:16:16:16, 64bpp, 16R, 16G, 16B, 16A, the 2-byte value for each R/G/B/A component is stored as little-endian"]
        RGBA64LE = ffi::AVPixelFormat_AV_PIX_FMT_RGBA64LE,
        #[doc = "< packed RGBA 16:16:16:16, 64bpp, 16B, 16G, 16R, 16A, the 2-byte value for each R/G/B/A component is stored as big-endian"]
        BGRA64BE = ffi::AVPixelFormat_AV_PIX_FMT_BGRA64BE,
        #[doc = "< packed RGBA 16:16:16:16, 64bpp, 16B, 16G, 16R, 16A, the 2-byte value for each R/G/B/A component is stored as little-endian"]
        BGRA64LE = ffi::AVPixelFormat_AV_PIX_FMT_BGRA64LE,
        #[doc = "< packed YUV 4:2:2, 16bpp, Y0 Cr Y1 Cb"]
        YVYU422 = ffi::AVPixelFormat_AV_PIX_FMT_YVYU422,
        #[doc = "< 16 bits gray, 16 bits alpha (big-endian)"]
        YA16BE = ffi::AVPixelFormat_AV_PIX_FMT_YA16BE,
        #[doc = "< 16 bits gray, 16 bits alpha (little-endian)"]
        YA16LE = ffi::AVPixelFormat_AV_PIX_FMT_YA16LE,
        #[doc = "< planar GBRA 4:4:4:4 32bpp"]
        GBRAP = ffi::AVPixelFormat_AV_PIX_FMT_GBRAP,
        #[doc = "< planar GBRA 4:4:4:4 64bpp, big-endian"]
        GBRAP16BE = ffi::AVPixelFormat_AV_PIX_FMT_GBRAP16BE,
        #[doc = "< planar GBRA 4:4:4:4 64bpp, little-endian"]
        GBRAP16LE = ffi::AVPixelFormat_AV_PIX_FMT_GBRAP16LE,
        #[doc = "  HW acceleration through QSV, data[3] contains a pointer to the"]
        #[doc = "  mfxFrameSurface1 structure."]
        QSV = ffi::AVPixelFormat_AV_PIX_FMT_QSV,
        #[doc = " HW acceleration though MMAL, data[3] contains a pointer to the"]
        #[doc = " MMAL_BUFFER_HEADER_T structure."]
        MMAL = ffi::AVPixelFormat_AV_PIX_FMT_MMAL,
        #[doc = "< HW decoding through Direct3D11 via old API, Picture.data[3] contains a ID3D11VideoDecoderOutputView pointer"]
        D3D11VA_VLD = ffi::AVPixelFormat_AV_PIX_FMT_D3D11VA_VLD,
        #[doc = " HW acceleration through CUDA. data[i] contain CUdeviceptr pointers"]
        #[doc = " exactly as for system memory frames."]
        CUDA = ffi::AVPixelFormat_AV_PIX_FMT_CUDA,
        #[doc = "< packed RGB 8:8:8, 32bpp, XRGBXRGB...   X=unused/undefined"]
        _0RGB = ffi::AVPixelFormat_AV_PIX_FMT_0RGB,
        #[doc = "< packed RGB 8:8:8, 32bpp, RGBXRGBX...   X=unused/undefined"]
        RGB0 = ffi::AVPixelFormat_AV_PIX_FMT_RGB0,
        #[doc = "< packed BGR 8:8:8, 32bpp, XBGRXBGR...   X=unused/undefined"]
        _0BGR = ffi::AVPixelFormat_AV_PIX_FMT_0BGR,
        #[doc = "< packed BGR 8:8:8, 32bpp, BGRXBGRX...   X=unused/undefined"]
        BGR0 = ffi::AVPixelFormat_AV_PIX_FMT_BGR0,
        #[doc = "< planar YUV 4:2:0,18bpp, (1 Cr & Cb sample per 2x2 Y samples), big-endian"]
        YUV420P12BE = ffi::AVPixelFormat_AV_PIX_FMT_YUV420P12BE,
        #[doc = "< planar YUV 4:2:0,18bpp, (1 Cr & Cb sample per 2x2 Y samples), little-endian"]
        YUV420P12LE = ffi::AVPixelFormat_AV_PIX_FMT_YUV420P12LE,
        #[doc = "< planar YUV 4:2:0,21bpp, (1 Cr & Cb sample per 2x2 Y samples), big-endian"]
        YUV420P14BE = ffi::AVPixelFormat_AV_PIX_FMT_YUV420P14BE,
        #[doc = "< planar YUV 4:2:0,21bpp, (1 Cr & Cb sample per 2x2 Y samples), little-endian"]
        YUV420P14LE = ffi::AVPixelFormat_AV_PIX_FMT_YUV420P14LE,
        #[doc = "< planar YUV 4:2:2,24bpp, (1 Cr & Cb sample per 2x1 Y samples), big-endian"]
        YUV422P12BE = ffi::AVPixelFormat_AV_PIX_FMT_YUV422P12BE,
        #[doc = "< planar YUV 4:2:2,24bpp, (1 Cr & Cb sample per 2x1 Y samples), little-endian"]
        YUV422P12LE = ffi::AVPixelFormat_AV_PIX_FMT_YUV422P12LE,
        #[doc = "< planar YUV 4:2:2,28bpp, (1 Cr & Cb sample per 2x1 Y samples), big-endian"]
        YUV422P14BE = ffi::AVPixelFormat_AV_PIX_FMT_YUV422P14BE,
        #[doc = "< planar YUV 4:2:2,28bpp, (1 Cr & Cb sample per 2x1 Y samples), little-endian"]
        YUV422P14LE = ffi::AVPixelFormat_AV_PIX_FMT_YUV422P14LE,
        #[doc = "< planar YUV 4:4:4,36bpp, (1 Cr & Cb sample per 1x1 Y samples), big-endian"]
        YUV444P12BE = ffi::AVPixelFormat_AV_PIX_FMT_YUV444P12BE,
        #[doc = "< planar YUV 4:4:4,36bpp, (1 Cr & Cb sample per 1x1 Y samples), little-endian"]
        YUV444P12LE = ffi::AVPixelFormat_AV_PIX_FMT_YUV444P12LE,
        #[doc = "< planar YUV 4:4:4,42bpp, (1 Cr & Cb sample per 1x1 Y samples), big-endian"]
        YUV444P14BE = ffi::AVPixelFormat_AV_PIX_FMT_YUV444P14BE,
        #[doc = "< planar YUV 4:4:4,42bpp, (1 Cr & Cb sample per 1x1 Y samples), little-endian"]
        YUV444P14LE = ffi::AVPixelFormat_AV_PIX_FMT_YUV444P14LE,
        #[doc = "< planar GBR 4:4:4 36bpp, big-endian"]
        GBRP12BE = ffi::AVPixelFormat_AV_PIX_FMT_GBRP12BE,
        #[doc = "< planar GBR 4:4:4 36bpp, little-endian"]
        GBRP12LE = ffi::AVPixelFormat_AV_PIX_FMT_GBRP12LE,
        #[doc = "< planar GBR 4:4:4 42bpp, big-endian"]
        GBRP14BE = ffi::AVPixelFormat_AV_PIX_FMT_GBRP14BE,
        #[doc = "< planar GBR 4:4:4 42bpp, little-endian"]
        GBRP14LE = ffi::AVPixelFormat_AV_PIX_FMT_GBRP14LE,
        #[doc = "< planar YUV 4:1:1, 12bpp, (1 Cr & Cb sample per 4x1 Y samples) full scale (JPEG), deprecated in favor of AV_PIX_FMT_YUV411P and setting color_range"]
        YUVJ411P = ffi::AVPixelFormat_AV_PIX_FMT_YUVJ411P,
        #[doc = "< bayer, BGBG..(odd line), GRGR..(even line), 8-bit samples */"]
        BAYER_BGGR8 = ffi::AVPixelFormat_AV_PIX_FMT_BAYER_BGGR8,
        #[doc = "< bayer, RGRG..(odd line), GBGB..(even line), 8-bit samples */"]
        BAYER_RGGB8 = ffi::AVPixelFormat_AV_PIX_FMT_BAYER_RGGB8,
        #[doc = "< bayer, GBGB..(odd line), RGRG..(even line), 8-bit samples */"]
        BAYER_GBRG8 = ffi::AVPixelFormat_AV_PIX_FMT_BAYER_GBRG8,
        #[doc = "< bayer, GRGR..(odd line), BGBG..(even line), 8-bit samples */"]
        BAYER_GRBG8 = ffi::AVPixelFormat_AV_PIX_FMT_BAYER_GRBG8,
        #[doc = "< bayer, BGBG..(odd line), GRGR..(even line), 16-bit samples, little-endian */"]
        BAYER_BGGR16LE = ffi::AVPixelFormat_AV_PIX_FMT_BAYER_BGGR16LE,
        #[doc = "< bayer, BGBG..(odd line), GRGR..(even line), 16-bit samples, big-endian */"]
        BAYER_BGGR16BE = ffi::AVPixelFormat_AV_PIX_FMT_BAYER_BGGR16BE,
        #[doc = "< bayer, RGRG..(odd line), GBGB..(even line), 16-bit samples, little-endian */"]
        BAYER_RGGB16LE = ffi::AVPixelFormat_AV_PIX_FMT_BAYER_RGGB16LE,
        #[doc = "< bayer, RGRG..(odd line), GBGB..(even line), 16-bit samples, big-endian */"]
        BAYER_RGGB16BE = ffi::AVPixelFormat_AV_PIX_FMT_BAYER_RGGB16BE,
        #[doc = "< bayer, GBGB..(odd line), RGRG..(even line), 16-bit samples, little-endian */"]
        BAYER_GBRG16LE = ffi::AVPixelFormat_AV_PIX_FMT_BAYER_GBRG16LE,
        #[doc = "< bayer, GBGB..(odd line), RGRG..(even line), 16-bit samples, big-endian */"]
        BAYER_GBRG16BE = ffi::AVPixelFormat_AV_PIX_FMT_BAYER_GBRG16BE,
        #[doc = "< bayer, GRGR..(odd line), BGBG..(even line), 16-bit samples, little-endian */"]
        BAYER_GRBG16LE = ffi::AVPixelFormat_AV_PIX_FMT_BAYER_GRBG16LE,
        #[doc = "< bayer, GRGR..(odd line), BGBG..(even line), 16-bit samples, big-endian */"]
        BAYER_GRBG16BE = ffi::AVPixelFormat_AV_PIX_FMT_BAYER_GRBG16BE,
        #[doc = "< XVideo Motion Acceleration via common packet passing"]
        XVMC = ffi::AVPixelFormat_AV_PIX_FMT_XVMC,
        #[doc = "< planar YUV 4:4:0,20bpp, (1 Cr & Cb sample per 1x2 Y samples), little-endian"]
        YUV440P10LE = ffi::AVPixelFormat_AV_PIX_FMT_YUV440P10LE,
        #[doc = "< planar YUV 4:4:0,20bpp, (1 Cr & Cb sample per 1x2 Y samples), big-endian"]
        YUV440P10BE = ffi::AVPixelFormat_AV_PIX_FMT_YUV440P10BE,
        #[doc = "< planar YUV 4:4:0,24bpp, (1 Cr & Cb sample per 1x2 Y samples), little-endian"]
        YUV440P12LE = ffi::AVPixelFormat_AV_PIX_FMT_YUV440P12LE,
        #[doc = "< planar YUV 4:4:0,24bpp, (1 Cr & Cb sample per 1x2 Y samples), big-endian"]
        YUV440P12BE = ffi::AVPixelFormat_AV_PIX_FMT_YUV440P12BE,
        #[doc = "< packed AYUV 4:4:4,64bpp (1 Cr & Cb sample per 1x1 Y & A samples), little-endian"]
        AYUV64LE = ffi::AVPixelFormat_AV_PIX_FMT_AYUV64LE,
        #[doc = "< packed AYUV 4:4:4,64bpp (1 Cr & Cb sample per 1x1 Y & A samples), big-endian"]
        AYUV64BE = ffi::AVPixelFormat_AV_PIX_FMT_AYUV64BE,
        #[doc = "< hardware decoding through Videotoolbox"]
        VIDEOTOOLBOX = ffi::AVPixelFormat_AV_PIX_FMT_VIDEOTOOLBOX,
        #[doc = "< like NV12, with 10bpp per component, data in the high bits, zeros in the low bits, little-endian"]
        P010LE = ffi::AVPixelFormat_AV_PIX_FMT_P010LE,
        #[doc = "< like NV12, with 10bpp per component, data in the high bits, zeros in the low bits, big-endian"]
        P010BE = ffi::AVPixelFormat_AV_PIX_FMT_P010BE,
        #[doc = "< planar GBR 4:4:4:4 48bpp, big-endian"]
        GBRAP12BE = ffi::AVPixelFormat_AV_PIX_FMT_GBRAP12BE,
        #[doc = "< planar GBR 4:4:4:4 48bpp, little-endian"]
        GBRAP12LE = ffi::AVPixelFormat_AV_PIX_FMT_GBRAP12LE,
        #[doc = "< planar GBR 4:4:4:4 40bpp, big-endian"]
        GBRAP10BE = ffi::AVPixelFormat_AV_PIX_FMT_GBRAP10BE,
        #[doc = "< planar GBR 4:4:4:4 40bpp, little-endian"]
        GBRAP10LE = ffi::AVPixelFormat_AV_PIX_FMT_GBRAP10LE,
        #[doc = "< hardware decoding through MediaCodec"]
        MEDIACODEC = ffi::AVPixelFormat_AV_PIX_FMT_MEDIACODEC,
        #[doc = "<        Y        , 12bpp, big-endian"]
        GRAY12BE = ffi::AVPixelFormat_AV_PIX_FMT_GRAY12BE,
        #[doc = "<        Y        , 12bpp, little-endian"]
        GRAY12LE = ffi::AVPixelFormat_AV_PIX_FMT_GRAY12LE,
        #[doc = "<        Y        , 10bpp, big-endian"]
        GRAY10BE = ffi::AVPixelFormat_AV_PIX_FMT_GRAY10BE,
        #[doc = "<        Y        , 10bpp, little-endian"]
        GRAY10LE = ffi::AVPixelFormat_AV_PIX_FMT_GRAY10LE,
        #[doc = "< like NV12, with 16bpp per component, little-endian"]
        P016LE = ffi::AVPixelFormat_AV_PIX_FMT_P016LE,
        #[doc = "< like NV12, with 16bpp per component, big-endian"]
        P016BE = ffi::AVPixelFormat_AV_PIX_FMT_P016BE,
        #[doc = " Hardware surfaces for Direct3D11."]
        #[doc = ""]
        #[doc = " This is preferred over the legacy AV_PIX_FMT_D3D11VA_VLD. The new D3D11"]
        #[doc = " hwaccel API and filtering support AV_PIX_FMT_D3D11 only."]
        #[doc = ""]
        #[doc = " data[0] contains a ID3D11Texture2D pointer, and data[1] contains the"]
        #[doc = " texture array index of the frame as intptr_t if the ID3D11Texture2D is"]
        #[doc = " an array texture (or always 0 if it's a normal texture)."]
        D3D11 = ffi::AVPixelFormat_AV_PIX_FMT_D3D11,
        #[doc = "<        Y        , 9bpp, big-endian"]
        GRAY9BE = ffi::AVPixelFormat_AV_PIX_FMT_GRAY9BE,
        #[doc = "<        Y        , 9bpp, little-endian"]
        GRAY9LE = ffi::AVPixelFormat_AV_PIX_FMT_GRAY9LE,
        #[doc = "< IEEE-754 single precision planar GBR 4:4:4,     96bpp, big-endian"]
        GBRPF32BE = ffi::AVPixelFormat_AV_PIX_FMT_GBRPF32BE,
        #[doc = "< IEEE-754 single precision planar GBR 4:4:4,     96bpp, little-endian"]
        GBRPF32LE = ffi::AVPixelFormat_AV_PIX_FMT_GBRPF32LE,
        #[doc = "< IEEE-754 single precision planar GBRA 4:4:4:4, 128bpp, big-endian"]
        GBRAPF32BE = ffi::AVPixelFormat_AV_PIX_FMT_GBRAPF32BE,
        #[doc = "< IEEE-754 single precision planar GBRA 4:4:4:4, 128bpp, little-endian"]
        GBRAPF32LE = ffi::AVPixelFormat_AV_PIX_FMT_GBRAPF32LE,
        #[doc = " DRM-managed buffers exposed through PRIME buffer sharing."]
        #[doc = ""]
        #[doc = " data[0] points to an AVDRMFrameDescriptor."]
        DRM_PRIME = ffi::AVPixelFormat_AV_PIX_FMT_DRM_PRIME,
        #[doc = " Hardware surfaces for OpenCL."]
        #[doc = ""]
        #[doc = " data[i] contain 2D image objects (typed in C as cl_mem, used"]
        #[doc = " in OpenCL as image2d_t) for each plane of the surface."]
        OPENCL = ffi::AVPixelFormat_AV_PIX_FMT_OPENCL,
        #[doc = "<        Y        , 14bpp, big-endian"]
        GRAY14BE = ffi::AVPixelFormat_AV_PIX_FMT_GRAY14BE,
        #[doc = "<        Y        , 14bpp, little-endian"]
        GRAY14LE = ffi::AVPixelFormat_AV_PIX_FMT_GRAY14LE,
        #[doc = "< IEEE-754 single precision Y, 32bpp, big-endian"]
        GRAYF32BE = ffi::AVPixelFormat_AV_PIX_FMT_GRAYF32BE,
        #[doc = "< IEEE-754 single precision Y, 32bpp, little-endian"]
        GRAYF32LE = ffi::AVPixelFormat_AV_PIX_FMT_GRAYF32LE,
        #[doc = "< planar YUV 4:2:2,24bpp, (1 Cr & Cb sample per 2x1 Y samples), 12b alpha, big-endian"]
        YUVA422P12BE = ffi::AVPixelFormat_AV_PIX_FMT_YUVA422P12BE,
        #[doc = "< planar YUV 4:2:2,24bpp, (1 Cr & Cb sample per 2x1 Y samples), 12b alpha, little-endian"]
        YUVA422P12LE = ffi::AVPixelFormat_AV_PIX_FMT_YUVA422P12LE,
        #[doc = "< planar YUV 4:4:4,36bpp, (1 Cr & Cb sample per 1x1 Y samples), 12b alpha, big-endian"]
        YUVA444P12BE = ffi::AVPixelFormat_AV_PIX_FMT_YUVA444P12BE,
        #[doc = "< planar YUV 4:4:4,36bpp, (1 Cr & Cb sample per 1x1 Y samples), 12b alpha, little-endian"]
        YUVA444P12LE = ffi::AVPixelFormat_AV_PIX_FMT_YUVA444P12LE,
        #[doc = "< planar YUV 4:4:4, 24bpp, 1 plane for Y and 1 plane for the UV components, which are interleaved (first byte U and the following byte V)"]
        NV24 = ffi::AVPixelFormat_AV_PIX_FMT_NV24,
        #[doc = "< as above, but U and V bytes are swapped"]
        NV42 = ffi::AVPixelFormat_AV_PIX_FMT_NV42
    }
}

enum_from_primitive! {
    #[derive(Debug, Copy, Clone, PartialEq)]
    #[repr(i32)]
    pub enum AVMediaType {
        Unknown = ffi::AVMediaType_AVMEDIA_TYPE_UNKNOWN,
        Video = ffi::AVMediaType_AVMEDIA_TYPE_VIDEO,
        Audio = ffi::AVMediaType_AVMEDIA_TYPE_AUDIO,
        Data = ffi::AVMediaType_AVMEDIA_TYPE_DATA,
        Subtitle = ffi::AVMediaType_AVMEDIA_TYPE_SUBTITLE,
        Attachment = ffi::AVMediaType_AVMEDIA_TYPE_ATTACHMENT
    }
}

enum_from_primitive! {
    #[doc = " Identify the syntax and semantics of the bitstream."]
    #[doc = " The principle is roughly:"]
    #[doc = " Two decoders with the same ID can decode the same streams."]
    #[doc = " Two encoders with the same ID can encode compatible streams."]
    #[doc = " There may be slight deviations from the principle due to implementation"]
    #[doc = " details."]
    #[allow(non_camel_case_types)]
    #[derive(Debug, Copy, Clone, PartialEq)]
    #[repr(u32)]
    pub enum AVCodecID {
        NONE = ffi::AVCodecID_AV_CODEC_ID_NONE,
        MPEG1VIDEO = ffi::AVCodecID_AV_CODEC_ID_MPEG1VIDEO,
        #[doc = "< preferred ID for MPEG-1/2 video decoding"]
        MPEG2VIDEO = ffi::AVCodecID_AV_CODEC_ID_MPEG2VIDEO,
        H261 = ffi::AVCodecID_AV_CODEC_ID_H261,
        H263 = ffi::AVCodecID_AV_CODEC_ID_H263,
        RV10 = ffi::AVCodecID_AV_CODEC_ID_RV10,
        RV20 = ffi::AVCodecID_AV_CODEC_ID_RV20,
        MJPEG = ffi::AVCodecID_AV_CODEC_ID_MJPEG,
        MJPEGB = ffi::AVCodecID_AV_CODEC_ID_MJPEGB,
        LJPEG = ffi::AVCodecID_AV_CODEC_ID_LJPEG,
        SP5X = ffi::AVCodecID_AV_CODEC_ID_SP5X,
        JPEGLS = ffi::AVCodecID_AV_CODEC_ID_JPEGLS,
        MPEG4 = ffi::AVCodecID_AV_CODEC_ID_MPEG4,
        RAWVIDEO = ffi::AVCodecID_AV_CODEC_ID_RAWVIDEO,
        MSMPEG4V1 = ffi::AVCodecID_AV_CODEC_ID_MSMPEG4V1,
        MSMPEG4V2 = ffi::AVCodecID_AV_CODEC_ID_MSMPEG4V2,
        MSMPEG4V3 = ffi::AVCodecID_AV_CODEC_ID_MSMPEG4V3,
        WMV1 = ffi::AVCodecID_AV_CODEC_ID_WMV1,
        WMV2 = ffi::AVCodecID_AV_CODEC_ID_WMV2,
        H263P = ffi::AVCodecID_AV_CODEC_ID_H263P,
        H263I = ffi::AVCodecID_AV_CODEC_ID_H263I,
        FLV1 = ffi::AVCodecID_AV_CODEC_ID_FLV1,
        SVQ1 = ffi::AVCodecID_AV_CODEC_ID_SVQ1,
        SVQ3 = ffi::AVCodecID_AV_CODEC_ID_SVQ3,
        DVVIDEO = ffi::AVCodecID_AV_CODEC_ID_DVVIDEO,
        HUFFYUV = ffi::AVCodecID_AV_CODEC_ID_HUFFYUV,
        CYUV = ffi::AVCodecID_AV_CODEC_ID_CYUV,
        H264 = ffi::AVCodecID_AV_CODEC_ID_H264,
        INDEO3 = ffi::AVCodecID_AV_CODEC_ID_INDEO3,
        VP3 = ffi::AVCodecID_AV_CODEC_ID_VP3,
        THEORA = ffi::AVCodecID_AV_CODEC_ID_THEORA,
        ASV1 = ffi::AVCodecID_AV_CODEC_ID_ASV1,
        ASV2 = ffi::AVCodecID_AV_CODEC_ID_ASV2,
        FFV1 = ffi::AVCodecID_AV_CODEC_ID_FFV1,
        _4XM = ffi::AVCodecID_AV_CODEC_ID_4XM,
        VCR1 = ffi::AVCodecID_AV_CODEC_ID_VCR1,
        CLJR = ffi::AVCodecID_AV_CODEC_ID_CLJR,
        MDEC = ffi::AVCodecID_AV_CODEC_ID_MDEC,
        ROQ = ffi::AVCodecID_AV_CODEC_ID_ROQ,
        INTERPLAY_VIDEO = ffi::AVCodecID_AV_CODEC_ID_INTERPLAY_VIDEO,
        XAN_WC3 = ffi::AVCodecID_AV_CODEC_ID_XAN_WC3,
        XAN_WC4 = ffi::AVCodecID_AV_CODEC_ID_XAN_WC4,
        RPZA = ffi::AVCodecID_AV_CODEC_ID_RPZA,
        CINEPAK = ffi::AVCodecID_AV_CODEC_ID_CINEPAK,
        WS_VQA = ffi::AVCodecID_AV_CODEC_ID_WS_VQA,
        MSRLE = ffi::AVCodecID_AV_CODEC_ID_MSRLE,
        MSVIDEO1 = ffi::AVCodecID_AV_CODEC_ID_MSVIDEO1,
        IDCIN = ffi::AVCodecID_AV_CODEC_ID_IDCIN,
        _8BPS = ffi::AVCodecID_AV_CODEC_ID_8BPS,
        SMC = ffi::AVCodecID_AV_CODEC_ID_SMC,
        FLIC = ffi::AVCodecID_AV_CODEC_ID_FLIC,
        TRUEMOTION1 = ffi::AVCodecID_AV_CODEC_ID_TRUEMOTION1,
        VMDVIDEO = ffi::AVCodecID_AV_CODEC_ID_VMDVIDEO,
        MSZH = ffi::AVCodecID_AV_CODEC_ID_MSZH,
        ZLIB = ffi::AVCodecID_AV_CODEC_ID_ZLIB,
        QTRLE = ffi::AVCodecID_AV_CODEC_ID_QTRLE,
        TSCC = ffi::AVCodecID_AV_CODEC_ID_TSCC,
        ULTI = ffi::AVCodecID_AV_CODEC_ID_ULTI,
        QDRAW = ffi::AVCodecID_AV_CODEC_ID_QDRAW,
        VIXL = ffi::AVCodecID_AV_CODEC_ID_VIXL,
        QPEG = ffi::AVCodecID_AV_CODEC_ID_QPEG,
        PNG = ffi::AVCodecID_AV_CODEC_ID_PNG,
        PPM = ffi::AVCodecID_AV_CODEC_ID_PPM,
        PBM = ffi::AVCodecID_AV_CODEC_ID_PBM,
        PGM = ffi::AVCodecID_AV_CODEC_ID_PGM,
        PGMYUV = ffi::AVCodecID_AV_CODEC_ID_PGMYUV,
        PAM = ffi::AVCodecID_AV_CODEC_ID_PAM,
        FFVHUFF = ffi::AVCodecID_AV_CODEC_ID_FFVHUFF,
        RV30 = ffi::AVCodecID_AV_CODEC_ID_RV30,
        RV40 = ffi::AVCodecID_AV_CODEC_ID_RV40,
        VC1 = ffi::AVCodecID_AV_CODEC_ID_VC1,
        WMV3 = ffi::AVCodecID_AV_CODEC_ID_WMV3,
        LOCO = ffi::AVCodecID_AV_CODEC_ID_LOCO,
        WNV1 = ffi::AVCodecID_AV_CODEC_ID_WNV1,
        AASC = ffi::AVCodecID_AV_CODEC_ID_AASC,
        INDEO2 = ffi::AVCodecID_AV_CODEC_ID_INDEO2,
        FRAPS = ffi::AVCodecID_AV_CODEC_ID_FRAPS,
        TRUEMOTION2 = ffi::AVCodecID_AV_CODEC_ID_TRUEMOTION2,
        BMP = ffi::AVCodecID_AV_CODEC_ID_BMP,
        CSCD = ffi::AVCodecID_AV_CODEC_ID_CSCD,
        MMVIDEO = ffi::AVCodecID_AV_CODEC_ID_MMVIDEO,
        ZMBV = ffi::AVCodecID_AV_CODEC_ID_ZMBV,
        AVS = ffi::AVCodecID_AV_CODEC_ID_AVS,
        SMACKVIDEO = ffi::AVCodecID_AV_CODEC_ID_SMACKVIDEO,
        NUV = ffi::AVCodecID_AV_CODEC_ID_NUV,
        KMVC = ffi::AVCodecID_AV_CODEC_ID_KMVC,
        FLASHSV = ffi::AVCodecID_AV_CODEC_ID_FLASHSV,
        CAVS = ffi::AVCodecID_AV_CODEC_ID_CAVS,
        JPEG2000 = ffi::AVCodecID_AV_CODEC_ID_JPEG2000,
        VMNC = ffi::AVCodecID_AV_CODEC_ID_VMNC,
        VP5 = ffi::AVCodecID_AV_CODEC_ID_VP5,
        VP6 = ffi::AVCodecID_AV_CODEC_ID_VP6,
        VP6F = ffi::AVCodecID_AV_CODEC_ID_VP6F,
        TARGA = ffi::AVCodecID_AV_CODEC_ID_TARGA,
        DSICINVIDEO = ffi::AVCodecID_AV_CODEC_ID_DSICINVIDEO,
        TIERTEXSEQVIDEO = ffi::AVCodecID_AV_CODEC_ID_TIERTEXSEQVIDEO,
        TIFF = ffi::AVCodecID_AV_CODEC_ID_TIFF,
        GIF = ffi::AVCodecID_AV_CODEC_ID_GIF,
        DXA = ffi::AVCodecID_AV_CODEC_ID_DXA,
        DNXHD = ffi::AVCodecID_AV_CODEC_ID_DNXHD,
        THP = ffi::AVCodecID_AV_CODEC_ID_THP,
        SGI = ffi::AVCodecID_AV_CODEC_ID_SGI,
        C93 = ffi::AVCodecID_AV_CODEC_ID_C93,
        BETHSOFTVID = ffi::AVCodecID_AV_CODEC_ID_BETHSOFTVID,
        PTX = ffi::AVCodecID_AV_CODEC_ID_PTX,
        TXD = ffi::AVCodecID_AV_CODEC_ID_TXD,
        VP6A = ffi::AVCodecID_AV_CODEC_ID_VP6A,
        AMV = ffi::AVCodecID_AV_CODEC_ID_AMV,
        VB = ffi::AVCodecID_AV_CODEC_ID_VB,
        PCX = ffi::AVCodecID_AV_CODEC_ID_PCX,
        SUNRAST = ffi::AVCodecID_AV_CODEC_ID_SUNRAST,
        INDEO4 = ffi::AVCodecID_AV_CODEC_ID_INDEO4,
        INDEO5 = ffi::AVCodecID_AV_CODEC_ID_INDEO5,
        MIMIC = ffi::AVCodecID_AV_CODEC_ID_MIMIC,
        RL2 = ffi::AVCodecID_AV_CODEC_ID_RL2,
        ESCAPE124 = ffi::AVCodecID_AV_CODEC_ID_ESCAPE124,
        DIRAC = ffi::AVCodecID_AV_CODEC_ID_DIRAC,
        BFI = ffi::AVCodecID_AV_CODEC_ID_BFI,
        CMV = ffi::AVCodecID_AV_CODEC_ID_CMV,
        MOTIONPIXELS = ffi::AVCodecID_AV_CODEC_ID_MOTIONPIXELS,
        TGV = ffi::AVCodecID_AV_CODEC_ID_TGV,
        TGQ = ffi::AVCodecID_AV_CODEC_ID_TGQ,
        TQI = ffi::AVCodecID_AV_CODEC_ID_TQI,
        AURA = ffi::AVCodecID_AV_CODEC_ID_AURA,
        AURA2 = ffi::AVCodecID_AV_CODEC_ID_AURA2,
        V210X = ffi::AVCodecID_AV_CODEC_ID_V210X,
        TMV = ffi::AVCodecID_AV_CODEC_ID_TMV,
        V210 = ffi::AVCodecID_AV_CODEC_ID_V210,
        DPX = ffi::AVCodecID_AV_CODEC_ID_DPX,
        MAD = ffi::AVCodecID_AV_CODEC_ID_MAD,
        FRWU = ffi::AVCodecID_AV_CODEC_ID_FRWU,
        FLASHSV2 = ffi::AVCodecID_AV_CODEC_ID_FLASHSV2,
        CDGRAPHICS = ffi::AVCodecID_AV_CODEC_ID_CDGRAPHICS,
        R210 = ffi::AVCodecID_AV_CODEC_ID_R210,
        ANM = ffi::AVCodecID_AV_CODEC_ID_ANM,
        BINKVIDEO = ffi::AVCodecID_AV_CODEC_ID_BINKVIDEO,
        IFF_ILBM = ffi::AVCodecID_AV_CODEC_ID_IFF_ILBM,
        KGV1 = ffi::AVCodecID_AV_CODEC_ID_KGV1,
        YOP = ffi::AVCodecID_AV_CODEC_ID_YOP,
        VP8 = ffi::AVCodecID_AV_CODEC_ID_VP8,
        PICTOR = ffi::AVCodecID_AV_CODEC_ID_PICTOR,
        ANSI = ffi::AVCodecID_AV_CODEC_ID_ANSI,
        A64_MULTI = ffi::AVCodecID_AV_CODEC_ID_A64_MULTI,
        A64_MULTI5 = ffi::AVCodecID_AV_CODEC_ID_A64_MULTI5,
        R10K = ffi::AVCodecID_AV_CODEC_ID_R10K,
        MXPEG = ffi::AVCodecID_AV_CODEC_ID_MXPEG,
        LAGARITH = ffi::AVCodecID_AV_CODEC_ID_LAGARITH,
        PRORES = ffi::AVCodecID_AV_CODEC_ID_PRORES,
        JV = ffi::AVCodecID_AV_CODEC_ID_JV,
        DFA = ffi::AVCodecID_AV_CODEC_ID_DFA,
        WMV3IMAGE = ffi::AVCodecID_AV_CODEC_ID_WMV3IMAGE,
        VC1IMAGE = ffi::AVCodecID_AV_CODEC_ID_VC1IMAGE,
        UTVIDEO = ffi::AVCodecID_AV_CODEC_ID_UTVIDEO,
        BMV_VIDEO = ffi::AVCodecID_AV_CODEC_ID_BMV_VIDEO,
        VBLE = ffi::AVCodecID_AV_CODEC_ID_VBLE,
        DXTORY = ffi::AVCodecID_AV_CODEC_ID_DXTORY,
        V410 = ffi::AVCodecID_AV_CODEC_ID_V410,
        XWD = ffi::AVCodecID_AV_CODEC_ID_XWD,
        CDXL = ffi::AVCodecID_AV_CODEC_ID_CDXL,
        XBM = ffi::AVCodecID_AV_CODEC_ID_XBM,
        ZEROCODEC = ffi::AVCodecID_AV_CODEC_ID_ZEROCODEC,
        MSS1 = ffi::AVCodecID_AV_CODEC_ID_MSS1,
        MSA1 = ffi::AVCodecID_AV_CODEC_ID_MSA1,
        TSCC2 = ffi::AVCodecID_AV_CODEC_ID_TSCC2,
        MTS2 = ffi::AVCodecID_AV_CODEC_ID_MTS2,
        CLLC = ffi::AVCodecID_AV_CODEC_ID_CLLC,
        MSS2 = ffi::AVCodecID_AV_CODEC_ID_MSS2,
        VP9 = ffi::AVCodecID_AV_CODEC_ID_VP9,
        AIC = ffi::AVCodecID_AV_CODEC_ID_AIC,
        ESCAPE130 = ffi::AVCodecID_AV_CODEC_ID_ESCAPE130,
        G2M = ffi::AVCodecID_AV_CODEC_ID_G2M,
        WEBP = ffi::AVCodecID_AV_CODEC_ID_WEBP,
        HNM4_VIDEO = ffi::AVCodecID_AV_CODEC_ID_HNM4_VIDEO,
        HEVC = ffi::AVCodecID_AV_CODEC_ID_HEVC,
        FIC = ffi::AVCodecID_AV_CODEC_ID_FIC,
        ALIAS_PIX = ffi::AVCodecID_AV_CODEC_ID_ALIAS_PIX,
        BRENDER_PIX = ffi::AVCodecID_AV_CODEC_ID_BRENDER_PIX,
        PAF_VIDEO = ffi::AVCodecID_AV_CODEC_ID_PAF_VIDEO,
        EXR = ffi::AVCodecID_AV_CODEC_ID_EXR,
        VP7 = ffi::AVCodecID_AV_CODEC_ID_VP7,
        SANM = ffi::AVCodecID_AV_CODEC_ID_SANM,
        SGIRLE = ffi::AVCodecID_AV_CODEC_ID_SGIRLE,
        MVC1 = ffi::AVCodecID_AV_CODEC_ID_MVC1,
        MVC2 = ffi::AVCodecID_AV_CODEC_ID_MVC2,
        HQX = ffi::AVCodecID_AV_CODEC_ID_HQX,
        TDSC = ffi::AVCodecID_AV_CODEC_ID_TDSC,
        HQ_HQA = ffi::AVCodecID_AV_CODEC_ID_HQ_HQA,
        HAP = ffi::AVCodecID_AV_CODEC_ID_HAP,
        DDS = ffi::AVCodecID_AV_CODEC_ID_DDS,
        DXV = ffi::AVCodecID_AV_CODEC_ID_DXV,
        SCREENPRESSO = ffi::AVCodecID_AV_CODEC_ID_SCREENPRESSO,
        RSCC = ffi::AVCodecID_AV_CODEC_ID_RSCC,
        AVS2 = ffi::AVCodecID_AV_CODEC_ID_AVS2,
        Y41P = ffi::AVCodecID_AV_CODEC_ID_Y41P,
        AVRP = ffi::AVCodecID_AV_CODEC_ID_AVRP,
        _012V = ffi::AVCodecID_AV_CODEC_ID_012V,
        AVUI = ffi::AVCodecID_AV_CODEC_ID_AVUI,
        AYUV = ffi::AVCodecID_AV_CODEC_ID_AYUV,
        TARGA_Y216 = ffi::AVCodecID_AV_CODEC_ID_TARGA_Y216,
        V308 = ffi::AVCodecID_AV_CODEC_ID_V308,
        V408 = ffi::AVCodecID_AV_CODEC_ID_V408,
        YUV4 = ffi::AVCodecID_AV_CODEC_ID_YUV4,
        AVRN = ffi::AVCodecID_AV_CODEC_ID_AVRN,
        CPIA = ffi::AVCodecID_AV_CODEC_ID_CPIA,
        XFACE = ffi::AVCodecID_AV_CODEC_ID_XFACE,
        SNOW = ffi::AVCodecID_AV_CODEC_ID_SNOW,
        SMVJPEG = ffi::AVCodecID_AV_CODEC_ID_SMVJPEG,
        APNG = ffi::AVCodecID_AV_CODEC_ID_APNG,
        DAALA = ffi::AVCodecID_AV_CODEC_ID_DAALA,
        CFHD = ffi::AVCodecID_AV_CODEC_ID_CFHD,
        TRUEMOTION2RT = ffi::AVCodecID_AV_CODEC_ID_TRUEMOTION2RT,
        M101 = ffi::AVCodecID_AV_CODEC_ID_M101,
        MAGICYUV = ffi::AVCodecID_AV_CODEC_ID_MAGICYUV,
        SHEERVIDEO = ffi::AVCodecID_AV_CODEC_ID_SHEERVIDEO,
        YLC = ffi::AVCodecID_AV_CODEC_ID_YLC,
        PSD = ffi::AVCodecID_AV_CODEC_ID_PSD,
        PIXLET = ffi::AVCodecID_AV_CODEC_ID_PIXLET,
        SPEEDHQ = ffi::AVCodecID_AV_CODEC_ID_SPEEDHQ,
        FMVC = ffi::AVCodecID_AV_CODEC_ID_FMVC,
        SCPR = ffi::AVCodecID_AV_CODEC_ID_SCPR,
        CLEARVIDEO = ffi::AVCodecID_AV_CODEC_ID_CLEARVIDEO,
        XPM = ffi::AVCodecID_AV_CODEC_ID_XPM,
        AV1 = ffi::AVCodecID_AV_CODEC_ID_AV1,
        BITPACKED = ffi::AVCodecID_AV_CODEC_ID_BITPACKED,
        MSCC = ffi::AVCodecID_AV_CODEC_ID_MSCC,
        SRGC = ffi::AVCodecID_AV_CODEC_ID_SRGC,
        SVG = ffi::AVCodecID_AV_CODEC_ID_SVG,
        GDV = ffi::AVCodecID_AV_CODEC_ID_GDV,
        FITS = ffi::AVCodecID_AV_CODEC_ID_FITS,
        IMM4 = ffi::AVCodecID_AV_CODEC_ID_IMM4,
        PROSUMER = ffi::AVCodecID_AV_CODEC_ID_PROSUMER,
        MWSC = ffi::AVCodecID_AV_CODEC_ID_MWSC,
        WCMV = ffi::AVCodecID_AV_CODEC_ID_WCMV,
        RASC = ffi::AVCodecID_AV_CODEC_ID_RASC,
        HYMT = ffi::AVCodecID_AV_CODEC_ID_HYMT,
        ARBC = ffi::AVCodecID_AV_CODEC_ID_ARBC,
        AGM = ffi::AVCodecID_AV_CODEC_ID_AGM,
        LSCR = ffi::AVCodecID_AV_CODEC_ID_LSCR,
        VP4 = ffi::AVCodecID_AV_CODEC_ID_VP4,
        PCM_S16LE = ffi::AVCodecID_AV_CODEC_ID_PCM_S16LE,
        PCM_S16BE = ffi::AVCodecID_AV_CODEC_ID_PCM_S16BE,
        PCM_U16LE = ffi::AVCodecID_AV_CODEC_ID_PCM_U16LE,
        PCM_U16BE = ffi::AVCodecID_AV_CODEC_ID_PCM_U16BE,
        PCM_S8 = ffi::AVCodecID_AV_CODEC_ID_PCM_S8,
        PCM_U8 = ffi::AVCodecID_AV_CODEC_ID_PCM_U8,
        PCM_MULAW = ffi::AVCodecID_AV_CODEC_ID_PCM_MULAW,
        PCM_ALAW = ffi::AVCodecID_AV_CODEC_ID_PCM_ALAW,
        PCM_S32LE = ffi::AVCodecID_AV_CODEC_ID_PCM_S32LE,
        PCM_S32BE = ffi::AVCodecID_AV_CODEC_ID_PCM_S32BE,
        PCM_U32LE = ffi::AVCodecID_AV_CODEC_ID_PCM_U32LE,
        PCM_U32BE = ffi::AVCodecID_AV_CODEC_ID_PCM_U32BE,
        PCM_S24LE = ffi::AVCodecID_AV_CODEC_ID_PCM_S24LE,
        PCM_S24BE = ffi::AVCodecID_AV_CODEC_ID_PCM_S24BE,
        PCM_U24LE = ffi::AVCodecID_AV_CODEC_ID_PCM_U24LE,
        PCM_U24BE = ffi::AVCodecID_AV_CODEC_ID_PCM_U24BE,
        PCM_S24DAUD = ffi::AVCodecID_AV_CODEC_ID_PCM_S24DAUD,
        PCM_ZORK = ffi::AVCodecID_AV_CODEC_ID_PCM_ZORK,
        PCM_S16LE_PLANAR = ffi::AVCodecID_AV_CODEC_ID_PCM_S16LE_PLANAR,
        PCM_DVD = ffi::AVCodecID_AV_CODEC_ID_PCM_DVD,
        PCM_F32BE = ffi::AVCodecID_AV_CODEC_ID_PCM_F32BE,
        PCM_F32LE = ffi::AVCodecID_AV_CODEC_ID_PCM_F32LE,
        PCM_F64BE = ffi::AVCodecID_AV_CODEC_ID_PCM_F64BE,
        PCM_F64LE = ffi::AVCodecID_AV_CODEC_ID_PCM_F64LE,
        PCM_BLURAY = ffi::AVCodecID_AV_CODEC_ID_PCM_BLURAY,
        PCM_LXF = ffi::AVCodecID_AV_CODEC_ID_PCM_LXF,
        S302M = ffi::AVCodecID_AV_CODEC_ID_S302M,
        PCM_S8_PLANAR = ffi::AVCodecID_AV_CODEC_ID_PCM_S8_PLANAR,
        PCM_S24LE_PLANAR = ffi::AVCodecID_AV_CODEC_ID_PCM_S24LE_PLANAR,
        PCM_S32LE_PLANAR = ffi::AVCodecID_AV_CODEC_ID_PCM_S32LE_PLANAR,
        PCM_S16BE_PLANAR = ffi::AVCodecID_AV_CODEC_ID_PCM_S16BE_PLANAR,
        PCM_S64LE = ffi::AVCodecID_AV_CODEC_ID_PCM_S64LE,
        PCM_S64BE = ffi::AVCodecID_AV_CODEC_ID_PCM_S64BE,
        PCM_F16LE = ffi::AVCodecID_AV_CODEC_ID_PCM_F16LE,
        PCM_F24LE = ffi::AVCodecID_AV_CODEC_ID_PCM_F24LE,
        PCM_VIDC = ffi::AVCodecID_AV_CODEC_ID_PCM_VIDC,
        ADPCM_IMA_QT = ffi::AVCodecID_AV_CODEC_ID_ADPCM_IMA_QT,
        ADPCM_IMA_WAV = ffi::AVCodecID_AV_CODEC_ID_ADPCM_IMA_WAV,
        ADPCM_IMA_DK3 = ffi::AVCodecID_AV_CODEC_ID_ADPCM_IMA_DK3,
        ADPCM_IMA_DK4 = ffi::AVCodecID_AV_CODEC_ID_ADPCM_IMA_DK4,
        ADPCM_IMA_WS = ffi::AVCodecID_AV_CODEC_ID_ADPCM_IMA_WS,
        ADPCM_IMA_SMJPEG = ffi::AVCodecID_AV_CODEC_ID_ADPCM_IMA_SMJPEG,
        ADPCM_MS = ffi::AVCodecID_AV_CODEC_ID_ADPCM_MS,
        ADPCM_4XM = ffi::AVCodecID_AV_CODEC_ID_ADPCM_4XM,
        ADPCM_XA = ffi::AVCodecID_AV_CODEC_ID_ADPCM_XA,
        ADPCM_ADX = ffi::AVCodecID_AV_CODEC_ID_ADPCM_ADX,
        ADPCM_EA = ffi::AVCodecID_AV_CODEC_ID_ADPCM_EA,
        ADPCM_G726 = ffi::AVCodecID_AV_CODEC_ID_ADPCM_G726,
        ADPCM_CT = ffi::AVCodecID_AV_CODEC_ID_ADPCM_CT,
        ADPCM_SWF = ffi::AVCodecID_AV_CODEC_ID_ADPCM_SWF,
        ADPCM_YAMAHA = ffi::AVCodecID_AV_CODEC_ID_ADPCM_YAMAHA,
        ADPCM_SBPRO_4 = ffi::AVCodecID_AV_CODEC_ID_ADPCM_SBPRO_4,
        ADPCM_SBPRO_3 = ffi::AVCodecID_AV_CODEC_ID_ADPCM_SBPRO_3,
        ADPCM_SBPRO_2 = ffi::AVCodecID_AV_CODEC_ID_ADPCM_SBPRO_2,
        ADPCM_THP = ffi::AVCodecID_AV_CODEC_ID_ADPCM_THP,
        ADPCM_IMA_AMV = ffi::AVCodecID_AV_CODEC_ID_ADPCM_IMA_AMV,
        ADPCM_EA_R1 = ffi::AVCodecID_AV_CODEC_ID_ADPCM_EA_R1,
        ADPCM_EA_R3 = ffi::AVCodecID_AV_CODEC_ID_ADPCM_EA_R3,
        ADPCM_EA_R2 = ffi::AVCodecID_AV_CODEC_ID_ADPCM_EA_R2,
        ADPCM_IMA_EA_SEAD = ffi::AVCodecID_AV_CODEC_ID_ADPCM_IMA_EA_SEAD,
        ADPCM_IMA_EA_EACS = ffi::AVCodecID_AV_CODEC_ID_ADPCM_IMA_EA_EACS,
        ADPCM_EA_XAS = ffi::AVCodecID_AV_CODEC_ID_ADPCM_EA_XAS,
        ADPCM_EA_MAXIS_XA = ffi::AVCodecID_AV_CODEC_ID_ADPCM_EA_MAXIS_XA,
        ADPCM_IMA_ISS = ffi::AVCodecID_AV_CODEC_ID_ADPCM_IMA_ISS,
        ADPCM_G722 = ffi::AVCodecID_AV_CODEC_ID_ADPCM_G722,
        ADPCM_IMA_APC = ffi::AVCodecID_AV_CODEC_ID_ADPCM_IMA_APC,
        ADPCM_VIMA = ffi::AVCodecID_AV_CODEC_ID_ADPCM_VIMA,
        ADPCM_AFC = ffi::AVCodecID_AV_CODEC_ID_ADPCM_AFC,
        ADPCM_IMA_OKI = ffi::AVCodecID_AV_CODEC_ID_ADPCM_IMA_OKI,
        ADPCM_DTK = ffi::AVCodecID_AV_CODEC_ID_ADPCM_DTK,
        ADPCM_IMA_RAD = ffi::AVCodecID_AV_CODEC_ID_ADPCM_IMA_RAD,
        ADPCM_G726LE = ffi::AVCodecID_AV_CODEC_ID_ADPCM_G726LE,
        ADPCM_THP_LE = ffi::AVCodecID_AV_CODEC_ID_ADPCM_THP_LE,
        ADPCM_PSX = ffi::AVCodecID_AV_CODEC_ID_ADPCM_PSX,
        ADPCM_AICA = ffi::AVCodecID_AV_CODEC_ID_ADPCM_AICA,
        ADPCM_IMA_DAT4 = ffi::AVCodecID_AV_CODEC_ID_ADPCM_IMA_DAT4,
        ADPCM_MTAF = ffi::AVCodecID_AV_CODEC_ID_ADPCM_MTAF,
        ADPCM_AGM = ffi::AVCodecID_AV_CODEC_ID_ADPCM_AGM,
        AMR_NB = ffi::AVCodecID_AV_CODEC_ID_AMR_NB,
        AMR_WB = ffi::AVCodecID_AV_CODEC_ID_AMR_WB,
        RA_144 = ffi::AVCodecID_AV_CODEC_ID_RA_144,
        RA_288 = ffi::AVCodecID_AV_CODEC_ID_RA_288,
        ROQ_DPCM = ffi::AVCodecID_AV_CODEC_ID_ROQ_DPCM,
        INTERPLAY_DPCM = ffi::AVCodecID_AV_CODEC_ID_INTERPLAY_DPCM,
        XAN_DPCM = ffi::AVCodecID_AV_CODEC_ID_XAN_DPCM,
        SOL_DPCM = ffi::AVCodecID_AV_CODEC_ID_SOL_DPCM,
        SDX2_DPCM = ffi::AVCodecID_AV_CODEC_ID_SDX2_DPCM,
        GREMLIN_DPCM = ffi::AVCodecID_AV_CODEC_ID_GREMLIN_DPCM,
        MP2 = ffi::AVCodecID_AV_CODEC_ID_MP2,
        #[doc = "< preferred ID for decoding MPEG audio layer 1, 2 or 3"]
        MP3 = ffi::AVCodecID_AV_CODEC_ID_MP3,
        AAC = ffi::AVCodecID_AV_CODEC_ID_AAC,
        AC3 = ffi::AVCodecID_AV_CODEC_ID_AC3,
        DTS = ffi::AVCodecID_AV_CODEC_ID_DTS,
        VORBIS = ffi::AVCodecID_AV_CODEC_ID_VORBIS,
        DVAUDIO = ffi::AVCodecID_AV_CODEC_ID_DVAUDIO,
        WMAV1 = ffi::AVCodecID_AV_CODEC_ID_WMAV1,
        WMAV2 = ffi::AVCodecID_AV_CODEC_ID_WMAV2,
        MACE3 = ffi::AVCodecID_AV_CODEC_ID_MACE3,
        MACE6 = ffi::AVCodecID_AV_CODEC_ID_MACE6,
        VMDAUDIO = ffi::AVCodecID_AV_CODEC_ID_VMDAUDIO,
        FLAC = ffi::AVCodecID_AV_CODEC_ID_FLAC,
        MP3ADU = ffi::AVCodecID_AV_CODEC_ID_MP3ADU,
        MP3ON4 = ffi::AVCodecID_AV_CODEC_ID_MP3ON4,
        SHORTEN = ffi::AVCodecID_AV_CODEC_ID_SHORTEN,
        ALAC = ffi::AVCodecID_AV_CODEC_ID_ALAC,
        WESTWOOD_SND1 = ffi::AVCodecID_AV_CODEC_ID_WESTWOOD_SND1,
        #[doc = "< as in Berlin toast format"]
        GSM = ffi::AVCodecID_AV_CODEC_ID_GSM,
        QDM2 = ffi::AVCodecID_AV_CODEC_ID_QDM2,
        COOK = ffi::AVCodecID_AV_CODEC_ID_COOK,
        TRUESPEECH = ffi::AVCodecID_AV_CODEC_ID_TRUESPEECH,
        TTA = ffi::AVCodecID_AV_CODEC_ID_TTA,
        SMACKAUDIO = ffi::AVCodecID_AV_CODEC_ID_SMACKAUDIO,
        QCELP = ffi::AVCodecID_AV_CODEC_ID_QCELP,
        WAVPACK = ffi::AVCodecID_AV_CODEC_ID_WAVPACK,
        DSICINAUDIO = ffi::AVCodecID_AV_CODEC_ID_DSICINAUDIO,
        IMC = ffi::AVCodecID_AV_CODEC_ID_IMC,
        MUSEPACK7 = ffi::AVCodecID_AV_CODEC_ID_MUSEPACK7,
        MLP = ffi::AVCodecID_AV_CODEC_ID_MLP,
        GSM_MS = ffi::AVCodecID_AV_CODEC_ID_GSM_MS,
        ATRAC3 = ffi::AVCodecID_AV_CODEC_ID_ATRAC3,
        APE = ffi::AVCodecID_AV_CODEC_ID_APE,
        NELLYMOSER = ffi::AVCodecID_AV_CODEC_ID_NELLYMOSER,
        MUSEPACK8 = ffi::AVCodecID_AV_CODEC_ID_MUSEPACK8,
        SPEEX = ffi::AVCodecID_AV_CODEC_ID_SPEEX,
        WMAVOICE = ffi::AVCodecID_AV_CODEC_ID_WMAVOICE,
        WMAPRO = ffi::AVCodecID_AV_CODEC_ID_WMAPRO,
        WMALOSSLESS = ffi::AVCodecID_AV_CODEC_ID_WMALOSSLESS,
        ATRAC3P = ffi::AVCodecID_AV_CODEC_ID_ATRAC3P,
        EAC3 = ffi::AVCodecID_AV_CODEC_ID_EAC3,
        SIPR = ffi::AVCodecID_AV_CODEC_ID_SIPR,
        MP1 = ffi::AVCodecID_AV_CODEC_ID_MP1,
        TWINVQ = ffi::AVCodecID_AV_CODEC_ID_TWINVQ,
        TRUEHD = ffi::AVCodecID_AV_CODEC_ID_TRUEHD,
        MP4ALS = ffi::AVCodecID_AV_CODEC_ID_MP4ALS,
        ATRAC1 = ffi::AVCodecID_AV_CODEC_ID_ATRAC1,
        BINKAUDIO_RDFT = ffi::AVCodecID_AV_CODEC_ID_BINKAUDIO_RDFT,
        BINKAUDIO_DCT = ffi::AVCodecID_AV_CODEC_ID_BINKAUDIO_DCT,
        AAC_LATM = ffi::AVCodecID_AV_CODEC_ID_AAC_LATM,
        QDMC = ffi::AVCodecID_AV_CODEC_ID_QDMC,
        CELT = ffi::AVCodecID_AV_CODEC_ID_CELT,
        G723_1 = ffi::AVCodecID_AV_CODEC_ID_G723_1,
        G729 = ffi::AVCodecID_AV_CODEC_ID_G729,
        _8SVX_EXP = ffi::AVCodecID_AV_CODEC_ID_8SVX_EXP,
        _8SVX_FIB = ffi::AVCodecID_AV_CODEC_ID_8SVX_FIB,
        BMV_AUDIO = ffi::AVCodecID_AV_CODEC_ID_BMV_AUDIO,
        RALF = ffi::AVCodecID_AV_CODEC_ID_RALF,
        IAC = ffi::AVCodecID_AV_CODEC_ID_IAC,
        ILBC = ffi::AVCodecID_AV_CODEC_ID_ILBC,
        OPUS = ffi::AVCodecID_AV_CODEC_ID_OPUS,
        COMFORT_NOISE = ffi::AVCodecID_AV_CODEC_ID_COMFORT_NOISE,
        TAK = ffi::AVCodecID_AV_CODEC_ID_TAK,
        METASOUND = ffi::AVCodecID_AV_CODEC_ID_METASOUND,
        PAF_AUDIO = ffi::AVCodecID_AV_CODEC_ID_PAF_AUDIO,
        ON2AVC = ffi::AVCodecID_AV_CODEC_ID_ON2AVC,
        DSS_SP = ffi::AVCodecID_AV_CODEC_ID_DSS_SP,
        CODEC2 = ffi::AVCodecID_AV_CODEC_ID_CODEC2,
        FFWAVESYNTH = ffi::AVCodecID_AV_CODEC_ID_FFWAVESYNTH,
        SONIC = ffi::AVCodecID_AV_CODEC_ID_SONIC,
        SONIC_LS = ffi::AVCodecID_AV_CODEC_ID_SONIC_LS,
        EVRC = ffi::AVCodecID_AV_CODEC_ID_EVRC,
        SMV = ffi::AVCodecID_AV_CODEC_ID_SMV,
        DSD_LSBF = ffi::AVCodecID_AV_CODEC_ID_DSD_LSBF,
        DSD_MSBF = ffi::AVCodecID_AV_CODEC_ID_DSD_MSBF,
        DSD_LSBF_PLANAR = ffi::AVCodecID_AV_CODEC_ID_DSD_LSBF_PLANAR,
        DSD_MSBF_PLANAR = ffi::AVCodecID_AV_CODEC_ID_DSD_MSBF_PLANAR,
        _4GV = ffi::AVCodecID_AV_CODEC_ID_4GV,
        INTERPLAY_ACM = ffi::AVCodecID_AV_CODEC_ID_INTERPLAY_ACM,
        XMA1 = ffi::AVCodecID_AV_CODEC_ID_XMA1,
        XMA2 = ffi::AVCodecID_AV_CODEC_ID_XMA2,
        DST = ffi::AVCodecID_AV_CODEC_ID_DST,
        ATRAC3AL = ffi::AVCodecID_AV_CODEC_ID_ATRAC3AL,
        ATRAC3PAL = ffi::AVCodecID_AV_CODEC_ID_ATRAC3PAL,
        DOLBY_E = ffi::AVCodecID_AV_CODEC_ID_DOLBY_E,
        APTX = ffi::AVCodecID_AV_CODEC_ID_APTX,
        APTX_HD = ffi::AVCodecID_AV_CODEC_ID_APTX_HD,
        SBC = ffi::AVCodecID_AV_CODEC_ID_SBC,
        ATRAC9 = ffi::AVCodecID_AV_CODEC_ID_ATRAC9,
        HCOM = ffi::AVCodecID_AV_CODEC_ID_HCOM,
        DVD_SUBTITLE = ffi::AVCodecID_AV_CODEC_ID_DVD_SUBTITLE,
        DVB_SUBTITLE = ffi::AVCodecID_AV_CODEC_ID_DVB_SUBTITLE,
        #[doc = "< raw UTF-8 text"]
        TEXT = ffi::AVCodecID_AV_CODEC_ID_TEXT,
        XSUB = ffi::AVCodecID_AV_CODEC_ID_XSUB,
        SSA = ffi::AVCodecID_AV_CODEC_ID_SSA,
        MOV_TEXT = ffi::AVCodecID_AV_CODEC_ID_MOV_TEXT,
        HDMV_PGS_SUBTITLE = ffi::AVCodecID_AV_CODEC_ID_HDMV_PGS_SUBTITLE,
        DVB_TELETEXT = ffi::AVCodecID_AV_CODEC_ID_DVB_TELETEXT,
        SRT = ffi::AVCodecID_AV_CODEC_ID_SRT,
        MICRODVD = ffi::AVCodecID_AV_CODEC_ID_MICRODVD,
        EIA_608 = ffi::AVCodecID_AV_CODEC_ID_EIA_608,
        JACOSUB = ffi::AVCodecID_AV_CODEC_ID_JACOSUB,
        SAMI = ffi::AVCodecID_AV_CODEC_ID_SAMI,
        REALTEXT = ffi::AVCodecID_AV_CODEC_ID_REALTEXT,
        STL = ffi::AVCodecID_AV_CODEC_ID_STL,
        SUBVIEWER1 = ffi::AVCodecID_AV_CODEC_ID_SUBVIEWER1,
        SUBVIEWER = ffi::AVCodecID_AV_CODEC_ID_SUBVIEWER,
        SUBRIP = ffi::AVCodecID_AV_CODEC_ID_SUBRIP,
        WEBVTT = ffi::AVCodecID_AV_CODEC_ID_WEBVTT,
        MPL2 = ffi::AVCodecID_AV_CODEC_ID_MPL2,
        VPLAYER = ffi::AVCodecID_AV_CODEC_ID_VPLAYER,
        PJS = ffi::AVCodecID_AV_CODEC_ID_PJS,
        ASS = ffi::AVCodecID_AV_CODEC_ID_ASS,
        HDMV_TEXT_SUBTITLE = ffi::AVCodecID_AV_CODEC_ID_HDMV_TEXT_SUBTITLE,
        TTML = ffi::AVCodecID_AV_CODEC_ID_TTML,
        ARIB_CAPTION = ffi::AVCodecID_AV_CODEC_ID_ARIB_CAPTION
    }
}

enum_from_primitive! {
    #[derive(Debug, Copy, Clone, PartialEq)]
    #[repr(i32)]
    pub enum AVDiscard {
        #[doc = "< discard nothing"]
        None = ffi::AVDiscard_AVDISCARD_NONE,
        #[doc = "< discard useless packets like 0 size packets in avi"]
        Default = ffi::AVDiscard_AVDISCARD_DEFAULT,
        #[doc = "< discard all non reference"]
        NonReference = ffi::AVDiscard_AVDISCARD_NONREF,
        #[doc = "< discard all bidirectional frames"]
        BiDirectional = ffi::AVDiscard_AVDISCARD_BIDIR,
        #[doc = "< discard all non intra frames"]
        NonIntra = ffi::AVDiscard_AVDISCARD_NONINTRA,
        #[doc = "< discard all frames except keyframes"]
        NonKey = ffi::AVDiscard_AVDISCARD_NONKEY,
        #[doc = "< discard all"]
        All = ffi::AVDiscard_AVDISCARD_ALL,
    }
}

enum_from_primitive! {
    #[derive(Debug, Copy, Clone, PartialEq)]
    #[repr(u32)]
    pub enum SwsScaler {
        FastBilinear = ffi::SWS_FAST_BILINEAR,
        Bilinear = ffi::SWS_BILINEAR,
        Bicubic = ffi::SWS_BICUBIC,
        X = ffi::SWS_X,
        Point = ffi::SWS_POINT,
        Area = ffi::SWS_AREA,
        Bicublin = ffi::SWS_BICUBLIN,
        Gauss = ffi::SWS_GAUSS,
        Sinc = ffi::SWS_SINC,
        Lanczos = ffi::SWS_LANCZOS,
        Spline = ffi::SWS_SPLINE,
    }
}