import {Genre} from "./models/Genre";
import {GenreWithContent} from "./models/dto/GenreWithContent";
import {Content} from "./models/Content";
import {ContentMeta} from "./models/dto/ContentMeta";
import {Instalment} from "./models/Instalment";
import {RequestErrorKind} from "../util/request/RequestErrorKind";
import {RequestError} from "../util/request/RequestError";

async function request<T>(
    baseUrl: URL,
    path: string,
    headers?: RequestInit
): Promise<T> {
    const url = new URL(path, baseUrl).toString();
    const response = await fetch(url, headers);
    if (response.ok) {
        return await response.json();
    } else if (response.status in RequestErrorKind) {
        throw new RequestError(response.statusText, response.status)
    } else {
        throw new RequestError(response.statusText, RequestErrorKind.UnknownError)
    }
}

export async function listGenres(baseUrl: URL): Promise<Genre[]> {
    return await request(baseUrl, `api/v1/genres`, {
        method: "GET"
    });
}

export async function getGenre(baseUrl: URL, genreId: string): Promise<GenreWithContent> {
    return await request(baseUrl, `api/v1/genres/${genreId}`, {
        method: "GET"
    });
}

export async function listContent(baseUrl: URL): Promise<Content[]> {
    return await request(baseUrl, `api/v1/content`, {
        method: "GET"
    });
}

export async function getContent(baseUrl: URL, contentId: string): Promise<ContentMeta> {
    return await request(baseUrl, `api/v1/content/${contentId}`, {
        method: "GET"
    });
}

export async function listEpisodes(baseUrl: URL, contentId: string): Promise<Instalment[]> {
    return await request(baseUrl, `api/v1/content/${contentId}/episodes`, {
        method: "GET"
    });
}
